function [t_array dt] = makeTime(kgrid, c, cfl)
%MAKETIME   Create an evenly spaced array of time points.
%
% DESCRIPTION:
%       makeTime creates an evenly spaced array of time points for use with
%       the first-order k-space simulation codes based on the
%       Courant-Friedrichs-Lewy stability level cfl and the grid size. The
%       time step dt is chosen based on cfl (the default setting is 0.3),
%       and the number of time steps is chosen based on the time it takes
%       to travel from one corner of the k-space grid given by kgrid to the
%       geometrically opposite corner travelling at c m/s. 
%
% USAGE:
%       [t_array dt] = makeTime(kgrid, c)
%       [t_array dt] = makeTime(kgrid, c, cfl)
%
% INPUTS:
%       kgrid       - kspace grid structure returned by makeGrid
%       c           - maximum sound speed in the medium [m/s]
%
% OPTIONAL INPUTS:
%       cfl         - Courant-Friedrichs-Lewy stability criterion 
%                     (default = 0.3) 
%
% OUTPUTS:
%       t_array     - array of evenly-spaced time points [s]
%       dt          - time step [s]
%
% ABOUT:
%       author      - Bradley Treeby
%       date        - 3rd July 2009
%       last update - 21st July 2009
%       
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
%
% See also makeGrid, kspaceFirstOrder1D, kspaceFirstOrder2D,
% kspaceFirstOrder3D 

% assign optional inputs
if nargin < 3
    % define the Courant-Friedrichs-Lewy stability level
    cfl = 0.3;
end

% if c is a matrix, take the maximum value
c = max(c(:));

% set the simulation length based on the size of the grid diagonal and
% the maximum sound speed in the medium
switch numDim(kgrid.k)
    case 1
        pixel_dim = kgrid.dx;
        t_end = kgrid.x_size/c;
    case 2
        pixel_dim = max([kgrid.dz, kgrid.dx]);
        t_end = sqrt(kgrid.z_size.^2 + kgrid.x_size.^2)/c;
    case 3
        pixel_dim = max([kgrid.dz, kgrid.dx, kgrid.dy]);
        t_end = sqrt(kgrid.z_size.^2 + kgrid.x_size.^2 + kgrid.y_size.^2)/c;
end
       
% assign a time step based on Courant stability criterion
dt = cfl*pixel_dim/c;

% create the time array
t_array = 0:dt:t_end;